# Make heatmap ----------------------------------------------------
# This will make grand heatmap
make_heatmap <- function(Data, no.time = TRUE){
  tryCatch({
    flog.debug("Calculating", name = log.name)
    

    if (no.time) {
      
      Data <- Data %>%
        filter(norm.time == 0)
      
    }
    
    Pheatmap.data <- Data %>% ungroup() %>%
      arrange(Event, norm.time) %>%
      unite(Event_time, norm.time, Event, sep = "__") %>%
      mutate(Direction_bin = case_when(Direction == "Up" ~ 1,
                                       Direction == "Down" ~ -1,
                                       Direction == "Not computed" ~ -2,
                                       TRUE ~ 0)) %>% 
      select(-Direction, -z.score, -Time.period) %>% unique()
    
    order <- unique(Pheatmap.data$Event_time)
    
    Pheatmap.data <- Pheatmap.data %>%
      spread(key = Event_time, value = Direction_bin)
    
    Pheatmap.data <- Pheatmap.data[,c("UnitID",order)]
    
    Pheatmap.data.mat <- Pheatmap.data %>% select(-UnitID) %>% as.matrix(.)
    row.names(Pheatmap.data.mat) <- Pheatmap.data$UnitID
  
    plot <- pheatmap(Pheatmap.data.mat,
                     cutree_rows = 3,
                     #cutree_cols = 2,
                     #clustering_distance_rows = "correlation",
                     scale = "none",
                     cluster_cols = FALSE,
                     clustering_method = "complete",
                     fontsize_row = 0,
                     fontsize_col = 10,
                     treeheight_row = 200,
                     show_rownames = FALSE,
                     silent = TRUE)[[4]]
    
    return(plot)
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}

# Make tilemap ----------------------------------------------------
# This will make a tile map with all of the unit Directions (to see where error are)
make_tilemap <- function(Data, no.time = TRUE){
  tryCatch({
    flog.debug("Calculating", name = log.name)

    if (no.time) {
      
      Data <- Data %>%
        filter(norm.time == 0)
      
    }
    
    Pheatmap.data <- Data %>% ungroup() %>%
      arrange(Event, norm.time) %>%
      unite(Event_time, norm.time, Event, sep = "__") %>%
      select(-z.score, -Time.period) %>% unique()
    
    plot <- ggplot(data = Pheatmap.data, aes(x = Event_time, y = UnitID, fill = Direction)) + 
      geom_tile() +
      theme_bw() + 
      scale_fill_manual(values = c("blue", "grey", "black", "red", "darkviolet")) + 
      theme(axis.text.x = element_text(angle = 90, hjust = 1))
    
    return(plot)
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}

# Make time plot ----------------------------------------------------
# This will save plot, needs ID to be used as file name, and location to save plot
single_unit_plot <- function(Plot){
  tryCatch({
    
    flog.debug("Making single unit plot", name = log.name)

    plot <- ggplot(data = Plot, 
                   aes(x = norm.time, y = z.score)) +
      geom_rect(aes(xmin = -2, xmax = -1, ymin = -Inf, ymax = Inf), fill = "grey", alpha = 0.05) +
      geom_rect(aes(xmin = 0, xmax = 1, ymin = -Inf, ymax = Inf), fill = "red", alpha = 0.01) +
      geom_line() + 
      facet_wrap(~UnitID) + 
      theme_bw() +
      geom_vline(xintercept = 0, linetype = "dashed", color = "grey50") +
      geom_hline(yintercept = 0, color = "grey50") +
      geom_text(aes(label = Direction, x = 7.5, y = max(z.score) * 0.8), size = 3, alpha = 0.5)
   
    return(plot)
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}

single_unit_perievent <- function(Plot){
  tryCatch({
    
    flog.debug("Making single unit plo for all perievents", name = log.name)
    
    plot <- ggplot(data = Plot, 
                   aes(x = norm.time, y = z.score)) +
      geom_rect(aes(xmin = -2, xmax = -1, ymin = -Inf, ymax = Inf), fill = "grey", alpha = 0.01) +
      geom_rect(aes(xmin = 0, xmax = 1, ymin = -Inf, ymax = Inf), fill = "slategrey", alpha = 0.001) +
      geom_vline(xintercept = 0, linetype = "dashed", color = "grey50") +
      geom_hline(yintercept = 0, color = "grey50") +
      geom_line(aes(color = Direction)) +
      geom_ribbon(aes(ymin = z.score - sem, ymax = z.score + sem), alpha = 0.1, size = 1) +
      theme_bw() +
      geom_text(aes(label = N, x = -5, y = max(z.score) * 0.6), size = 3, alpha = 0.5) +
      geom_text(aes(label = Direction, x = 7.5, y = max(z.score) * 0.6), size = 3, alpha = 0.5) + 
      facet_wrap(~Event, scales = "free_y") +
      theme(legend.position = "none")
    
    return(plot)
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}


# Make population time plot ----------------------------------------------------
# This will save plot, needs ID to be used as file name, and location to save plot
population_plot <- function(Plot, by.direction = FALSE){
  tryCatch({
    
    flog.debug("Making population plot", name = log.name)
    
    plot <- ggplot(data = Plot, 
                   aes(x = norm.time, y = z.score)) +
      geom_rect(aes(xmin = -2, xmax = -1, ymin = -Inf, ymax = Inf), fill = "grey", alpha = 0.01) +
      geom_rect(aes(xmin = 0, xmax = 1, ymin = -Inf, ymax = Inf), fill = "slategrey", alpha = 0.001) +
      geom_vline(xintercept = 0, linetype = "dashed", color = "grey50") +
      geom_hline(yintercept = 0, color = "grey50") +
      geom_line(color = "black") +
      geom_ribbon(aes(ymin = z.score - sem, ymax = z.score + sem), alpha = 0.1, size = 1) +
      geom_line(aes(y = 0, color = T.val), size = 6) +
      theme_bw() +
      scale_color_distiller(palette = "RdYlBu", na.value = NA) +
      geom_text(aes(label = N, x = 10, y = max(z.score) * 0.6), size = 3, alpha = 0.5)
    
    if (by.direction) {
      plot <- plot + facet_grid(Direction~Event, scales = "free_y")
      
    } else {
      plot <- plot + facet_wrap(~Event, scales = "free_y")
    }
    
    return(plot)
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}


# Make bar plot ----------------------------------------------------
# TMake a bar plot to show Unit distribution
distribution_plot <- function(Plot){
  tryCatch({
    
    flog.debug("Making bar plot", name = log.name)
    
    plot <- ggplot(Plot, aes(Event, perc, fill = Direction)) + 
      facet_grid(TreatmentID ~ Area.brain) + 
      geom_bar(stat = "identity") +
      theme_bw() + 
      theme(axis.text.x = element_text(angle = 90, hjust = 1, vjust = -0.001))

    if (n_distinct(Plot$Direction) > 3) {
      plot <- plot + scale_fill_manual(values = c("blue", "grey", "black", "red", "darkviolet"))
    } else {
      plot <- plot + scale_fill_manual(values = c("blue", "grey", "red"))
    }

    return(plot)
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}


# scatter.plt ----------------------------------------------------
# scatterplot for firing
scatter_plot <- function(Plot, x_axis = "avg.pokes", y_axis = "z.score", 
                         color_shape = "Direction", grouping = "Direction",
                         xlabel = "Precue response rate", ylabel = "Z score"){
  tryCatch({
    
    flog.debug("Making scatter plot", name = log.name)
    
    plot <- ggplot(data = Plot,
                   aes_string(x = x_axis, y = y_axis, color = color_shape, shape = color_shape, group = grouping)) + 
      geom_point() + 
      geom_smooth(method = "lm") +
      ylab(ylabel) +
      xlab(xlabel) +
      theme_bw() + 
      theme(panel.background = element_rect(fill = "white"),
            strip.background = element_blank(),
            axis.text.y = element_text(face = "bold", size = 12),
            axis.text.x = element_text(face = "bold", size = 16),
            axis.title =  element_text(face = "bold", size = 28),
            strip.text.x = element_text(face = "bold", size = 28),
            strip.text.y = element_text(face = "bold", size = 28),
            panel.grid.major = element_blank(),
            panel.border = element_rect(colour = "black")) +
      theme(text = element_text(family = "sans")) + 
      facet_grid(~Direction)
    
    return(plot)
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}

# Save plot data ----------------------------------------------------
# This will save plot, needs ID to be used as file name, and location to save plot
save_plot <- function(Plot, ID, plots.folder, width = 10, height = 20){
  tryCatch({
    
    flog.debug("Saving plot", name = log.name)
    flog.debug(paste0("ID: ", ID), name = log.name)
    
    if (is_empty(Data)) {
      flog.warn("Plot data empty!", name = log.name)
      
    } else {
      
      ggsave(filename = paste0(plots.folder, ID, ".png"), 
             plot = Plot, 
             type = "cairo-png",
             dpi = 300,
             width = width, 
             height = height)
      
      stopifnot(!is.na(Plot))
      
    }
    
  },
  error = function(c){
    flog.error("ERROR!", name = log.name)
    flog.error(c, name = log.name)
    return(NULL)
  }) # end try catch   
}






